/*Copyright (c) 2012, Sierra Wireless, Inc
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:
Redistributions of source code must retain the above copyright notice, this
 list of conditions and the following disclaimer.
Redistributions in binary form must reproduce the above copyright notice, this
 list of conditions and the following disclaimer in the documentation and/or
 other materials provided with the distribution.
Neither the name of Sierra Wireless Inc, Richmond, Canada nor the names of its
 contributors may be used to endorse or promote products derived from this
 software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABIL ITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

/***************************************************************************/
/*  File       : GPRS_management.c
 *
 *  Object     : GPRS bearer configuration:
 *
 *  [GPRS_APN]:      Network provider APN
 *  [GPRS_USER]:     login name
 *  [GPRS_PASSWORD]: password
 *  [GPRS_PINCODE]:  PIN code of the SIM card, or NULL if not required
 */
/***************************************************************************/


/*****************************************************************************/
/*  Header files                                                             */
/*****************************************************************************/

#include "adl_global.h"
#include "adl_UnSoHandler.h"
#include "wip.h"

/***************************************************************************/
/*  Defines                                                                */
/***************************************************************************/
#define GPRS_APN        "orangeinternet"
#define GPRS_USER       ""
#define GPRS_PASSWORD   ""
#define GPRS_PINCODE    "0000"
#define GPRS_BEARER     "GPRS"

#define REG_STATE_STOPPED	0
#define REG_STATE_REG   	1
#define REG_STATE_SEARCHING 2
#define REG_STATE_ROAM  	5

/* in 100ms steps */
#define CREG_POLLING_PERIOD     20

#define NORMAL_TRACE_LEVEL      4
#define ERROR_TRACE_LEVEL       1

#define ASSERT( pred )  if( !(pred)) TRACE(( ERROR_TRACE_LEVEL, "ASSERTION FAILURE line %i: " #pred "\n", __LINE__))
#define ASSERT_OK( v )  ASSERT( OK == (v))


/*****************************************************************************/
/*  Prototypes                                                               */
/*****************************************************************************/

/* Function to be called once the bearer is up and running. */
static void ( * AppliEntryPoint) ( void ) ;

static void OpenAndStartBearer ( void ) ;

/*****************************************************************************/
/*  Initialization-related event handlers                                    */
/*****************************************************************************/
static void cbEvhBearer ( wip_bearer_t b, s8 event, void *ctx ) ;

/*****************************************************************************/
/*  Function   : cbEvhBearer                                                 */
/*---------------------------------------------------------------------------*/
/*  Object     : bearer events handler: when the bearer connection is        */
/*               completed, start IP services                                */
/*                                                                           */
/*  Return     : None                                                        */
/*---------------------------------------------------------------------------*/
/*  Variable Name     |IN |OUT|GLB|  Utilization                             */
/*--------------------+---+---+---+------------------------------------------*/
/*  b                 | X |   |   | bearer identifier                        */
/*--------------------+---+---+---+------------------------------------------*/
/*  event             | X |   |   | bearer event type                        */
/*--------------------+---+---+---+------------------------------------------*/
/*  ctx               |   |   |   | unused                                   */
/*--------------------+---+---+---+------------------------------------------*/
/*****************************************************************************/
static void cbEvhBearer ( wip_bearer_t b, s8 event, void *ctx )
{
    if ( WIP_BEV_IP_CONNECTED == event )
    {
        AppliEntryPoint ( ) ;
    }
}

/*****************************************************************************/
/*  Function   : OpenAndStartBearer()                                        */
/*---------------------------------------------------------------------------*/
/*  Object : Open and start the GPRS bearer. Normally, the bearer will       */
/*           answer IN_PROGRESS, and the initialization will be finished     */
/*           by the callback cbEvhBearer.                                    */
/*                                                                           */
/*  Return : None                                                            */
/*---------------------------------------------------------------------------*/
/*  Variable Name     |IN |OUT|GLB|  Utilization                             */
/*--------------------+---+---+---+------------------------------------------*/
/*--------------------+---+---+---+------------------------------------------*/
/*****************************************************************************/
static void OpenAndStartBearer ( void )
{
    s8 bearerRet;
    wip_bearer_t br;

    /* Open the GPRS bearer */
    bearerRet = wip_bearerOpen ( &br, GPRS_BEARER, cbEvhBearer, NULL ) ;
    ASSERT_OK( bearerRet);

    /* Set the bearer configuration */
    bearerRet = wip_bearerSetOpts ( br,
                            WIP_BOPT_GPRS_APN, GPRS_APN,
                            WIP_BOPT_LOGIN, GPRS_USER,
                            WIP_BOPT_PASSWORD, GPRS_PASSWORD,
                            WIP_BOPT_END );
    ASSERT_OK( bearerRet );

    /* Start the bearer */
    bearerRet = wip_bearerStart ( br );
    ASSERT( OK == bearerRet || WIP_BERR_OK_INPROGRESS == bearerRet);
}


/*****************************************************************************/
/*  Function   : Unsolicited CREG handler                                                     */
/*---------------------------------------------------------------------------*/
/*  Object     : initialize GPRS connection, then launch EntryPoint() on     */
/*               success.                                                    */
/*                                                                           */
/*  Return     : None                                                        */
/*---------------------------------------------------------------------------*/
/*  Variable Name     |IN |OUT|GLB|  Utilization                             */
/*--------------------+---+---+---+------------------------------------------*/
/*  EntryPoint        | X |   |   |Function run after successful connection  */
/*--------------------+---+---+---+------------------------------------------*/
/*****************************************************************************/

bool CREG_handler(adl_atUnsolicited_t *paras)
{
    ascii *rsp;
    ascii regStateString [ 3 ];
    s32 regStateInt;

    TRACE ( ( NORMAL_TRACE_LEVEL, "Entered CREG state change handler" ) ) ;

    rsp = ( ascii * ) adl_memGet ( paras->StrLength );
    wm_strRemoveCRLF ( rsp, paras->StrData, paras->StrLength );

    wm_strGetParameterString ( regStateString, paras->StrData, 1 );	//Parse out parameter 1 of the +CREG response as detailed in AT command manual
    regStateInt = wm_atoi ( regStateString );

    TRACE ( ( NORMAL_TRACE_LEVEL, "Parsed state, %d", regStateInt ) ) ;

    switch (regStateInt)
    {
    	case REG_STATE_REG: case REG_STATE_ROAM:
				TRACE ((NORMAL_TRACE_LEVEL, "Registered on GSM network, state %d", regStateInt));	/* Registration is complete so open and start bearer */
				OpenAndStartBearer ( ) ;
				break;
    }
    return TRUE;
}

/*****************************************************************************/
/*  Function   : CfgGprs                                                     */
/*---------------------------------------------------------------------------*/
/*  Object     : initialize GPRS connection, then launch EntryPoint() on     */
/*               success.                                                    */
/*                                                                           */
/*  Return     : None                                                        */
/*---------------------------------------------------------------------------*/
/*  Variable Name     |IN |OUT|GLB|  Utilization                             */
/*--------------------+---+---+---+------------------------------------------*/
/*  EntryPoint        | X |   |   |Function run after successful connection  */
/*--------------------+---+---+---+------------------------------------------*/
/*****************************************************************************/
void CfgGprs ( void ( * EntryPoint ) ( void ) )
{
    TRACE ( ( NORMAL_TRACE_LEVEL, "(CfgGprs) Enter." ) ) ;
    AppliEntryPoint = EntryPoint;
//    adl_simSubscribe ( cbEvhSim, GPRS_PINCODE );	//An alternative way to start the connection off is to wait for SIM initialization
    adl_atUnSoSubscribe("+CREG", (adl_atUnSoHandler_t) CREG_handler);	//Will call CREG_handler each time the GSM state (NOT GPRS) state changes
}

