/*Copyright (c) 2012, Sierra Wireless, Inc
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:
Redistributions of source code must retain the above copyright notice, this
 list of conditions and the following disclaimer.
Redistributions in binary form must reproduce the above copyright notice, this
 list of conditions and the following disclaimer in the documentation and/or
 other materials provided with the distribution.
Neither the name of Sierra Wireless Inc, Richmond, Canada nor the names of its
 contributors may be used to endorse or promote products derived from this
 software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABIL ITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

/**********************************************************************************************************************/
/*  IP management.c                                                                                                   */
/*                                                                                                                    */
/*                                                                                                                    */
/**********************************************************************************************************************/

/****************************************************************************
 * File    :   entry_point.c                                                        
 *
 * Very basic TCP client application:
 * - Connects a TCP client socket to google.com
 *   on port 80]
 * - Sends strings to request the front page
 * - Puts everything received by the socket into [rcv_buffer], until
 *   the table is full.
 */

/*****************************************************************************/
/*  Include files                                                            */
/*****************************************************************************/

#include "adl_global.h"
#include "wip.h"

/***************************************************************************/
/*  Defines                                                                */
/***************************************************************************/

#define TCP_PEER_GOOGLE        "www.google.com"
#define TCP_PEER_PORT           80

#define RCV_BUFFER_SIZE         10240

#define NORMAL_TRACE_LEVEL      4
#define ERROR_TRACE_LEVEL       1


/***************************************************************************/
/*  Globals                                                                */
/***************************************************************************/

/* HTML string to ask Google to download its front page */
static u8 snd_buffer [ ] = { "get /index.html HTTP/1.1\nHost:www.google.co.uk\n\n" };	//Google string

/* How many bytes of [buffer] have already been sent. */
static int snd_offset = 0;

static char rcv_buffer [ RCV_BUFFER_SIZE ];
static int rcv_offset = 0;

/***************************************************************************/
/*  Function prototypes                                                    */
/***************************************************************************/
static void cbevh ( wip_event_t *ev, void *ctx );

/***************************************************************************/
/*  Function   : AppliEntryPoint                                           */
/*-------------------------------------------------------------------------*/
/*  Object     : Called once the WIP IP stack is fully initialized.        */
/*               This is the starting point of user applications.          */
/*-------------------------------------------------------------------------*/
/***************************************************************************/
void AppliEntryPoint ( void )
{
    wip_channel_t socket;

    TRACE ( ( NORMAL_TRACE_LEVEL, "[SAMPLE]: connecting to client %s:%i", TCP_PEER_GOOGLE, TCP_PEER_PORT ) );
    socket = wip_TCPClientCreate ( TCP_PEER_GOOGLE, TCP_PEER_PORT, cbevh, NULL );
    if ( !socket )
    {
        TRACE ( ( ERROR_TRACE_LEVEL, "[SAMPLE] Can't connect" ) );
        return;
    }
}

/***************************************************************************/
/*  Function   : cbevh                                                     */
/*-------------------------------------------------------------------------*/
/*  Object     : Handling events happenning on the TCP client socket.      */
/*-------------------------------------------------------------------------*/
/*  Variable Name     |IN |OUT|GLB|  Utilisation                           */
/*--------------------+---+---+---+----------------------------------------*/
/*  ev                | X |   |   |  WIP event                             */
/*--------------------+---+---+---+----------------------------------------*/
/*  ctx               | X |   |   |  user data (unused)                    */
/*--------------------+---+---+---+----------------------------------------*/
/***************************************************************************/
static void cbevh ( wip_event_t *ev, void *ctx )
{
    switch ( ev->kind )
    {
        case WIP_CEV_OPEN:
        {
            TRACE ( ( NORMAL_TRACE_LEVEL, "[SAMPLE] Connection established successfully" ) );
            break;
        }

        case WIP_CEV_READ:	//Typically Google front page is larger than buffer, in this case further data is ignored until the application deals with what is currently in the buffer
        {
            int nread;
            TRACE ( ( NORMAL_TRACE_LEVEL, "[SAMPLE] Some data arrived\n" ) );
            nread = wip_read ( ev->channel, rcv_buffer + rcv_offset, sizeof ( rcv_buffer ) - rcv_offset );
            if ( nread )
            {
                adl_atSendResponse ( ADL_AT_UNS, "\r\nTCP Client: Data Received\r\n" );
            }
            if ( nread < 0 )
            {
                TRACE ( ( ERROR_TRACE_LEVEL, "[SAMPLE] read error %i", nread ) );
                return;
            }
            rcv_offset += nread;
            if ( rcv_offset == sizeof ( rcv_buffer ) )
            {
                TRACE ( ( ERROR_TRACE_LEVEL, "[SAMPLE] Reception capacity exceeded, won't read more" ) );
            }
            else
            {
                TRACE ( ( NORMAL_TRACE_LEVEL, "[SAMPLE] Wrote %i bytes of data from network to rcv_buffer. "
                                "%i bytes remain available in rcv_buffer\n", nread, sizeof ( rcv_buffer ) - rcv_offset ) );
            }
            break;
        }

        case WIP_CEV_WRITE:
        {
            int nwrite;
            TRACE ( ( NORMAL_TRACE_LEVEL, "[SAMPLE] Can send more data" ) );
            nwrite = wip_write ( ev->channel, snd_buffer + snd_offset, sizeof ( snd_buffer ) - snd_offset );
			if ( nwrite )
            {
                adl_atSendResponse ( ADL_AT_UNS, "\r\nTCP Client: Data sent\r\n" );
            }
            if ( nwrite < 0 )
            {
                TRACE ( ( ERROR_TRACE_LEVEL, "[SAMPLE] write error %i", nwrite ) );
                return;
            }
            snd_offset += nwrite;
            if ( snd_offset == sizeof ( snd_buffer ) )
            {
                TRACE ( ( NORMAL_TRACE_LEVEL, "[SAMPLE] Everything has been sent, won't send more." ) );
            }
            else
            {
                TRACE ( ( NORMAL_TRACE_LEVEL, "[SAMPLE] Wrote %i bytes. %i bytes left to send in snd_buffer",
                                nwrite, sizeof ( snd_buffer ) - snd_offset ) );
            }
            break;
        }

        case WIP_CEV_ERROR:
        {
            TRACE ( ( ERROR_TRACE_LEVEL, "[SAMPLE] Error %i on socket. Closing.", ev->content.error.errnum ) );
            wip_close ( ev->channel );
            break;
        }

        case WIP_CEV_PEER_CLOSE:
        {
            TRACE ( ( NORMAL_TRACE_LEVEL, "[SAMPLE] Connection closed by peer" ) );
            wip_close ( ev->channel );
            break;
        }

        default:
        {
            break;
        }
    }
}

